/*
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 */
 
package soporte.html;
/**
 * HTML.java
 * This class has been crated to facilitate the creation of
 * HTML tags. The result of the public methods is a String 
 * containing a HTML tag or attibute. This class is still under construction,
 * so it might contains errors and its set if HTML tags is not complete.
 *
 * Created: Sun Aug 24 21:34:52 2003
 *
 * @author <a href="mailto:llana@sip.ucm.es">Luis Fernando Llana Díaz</a>
 * @version $Id: Tags.java,v 1.26 2006-09-07 12:29:26 luis Exp $
 */
import java.io.*;
import java.util.*;

/**
 * Describe class <code>Tags</code> here.
 *
 * @author <a href="mailto:luis@localhost">Luis Fernando Llana Díaz</a>
 * @version $Id: Tags.java,v 1.26 2006-09-07 12:29:26 luis Exp $
 */
public class Tags {

    /**
     * The formMethod variable contains the default form method (post or get).
     * It is static, so it must be used carefully and consistently. It should
     * always have the same value along the application or it .
     */
    private static String formMethod="get";

    /**
     * Get the <code>FormMethod</code> value. 
     * 
     *
     * @return a <code>String</code> value
     */
    public static final String getFormMethod() {
	return formMethod;
    }

    /**
     * Set the <code>FormMethod</code> value.
     *
     * @param newFormMethod The new FormMethod value. 
     * That variable is static, so it must be used carefully and consistently. It should
     * always have the same value along the application or it .
     */
    public static final void setFormMethod(final String newFormMethod) {
	formMethod = newFormMethod;
    }
    /**
     * <code>strOnChange</code> is used for <em>auto submit</em> input HTML tags.
     *
     */
    public static final String strOnChange=" onChange=\"enviarFormulario(this)\" ";
    
    
    /**
     *  <code>strSubmit</code> is the name of submit buttons.
     *
     */
    public static final String strSubmit="submit";
    
    
    /**
     * The <code>tag</code> method returns a generic tag, with beginnig and end tags.
     * It can also be used to generate any XML tag with contents. If the contents is empty
     * and writeEmpty is false 
     * the method returns an empty string.
     *
     * @param name a <code>String</code> value containing the name of the tag
     * @param attr a <code>String</code> value containing the tag attributes.
     * @param contents a <code>String</code> value with the contents of the tag.     
     * No checking of the code is carried out.     
     * @param writeEmpty is a <code>Boolean</code>, if true the tag is written regardless
     * the content is empty or not. If it is false and the contents is empty, the empty 
     * string is returned.
     * @return a <code>String</code> value with the tag.
     */
    public static String tag(String name, String attr, String contents, boolean writeEmpty) {
	if (writeEmpty || !isEmpty(contents)) {
	    return "<"+name+" "+attr+">"+contents+"</"+name+">\n";
	} else {
	    return "";
	}
    }
    
    /**
     * The <code>tag</code> method returns a generic tag, with beginnig and end tags.
     * It can also be used to generate any XML tag with contents. If the contents is empty, 
     * the method returns an empty string. It is exactly the same as
     * <code>tag(name,attr,contents,false)</code>.
     *
     * @param name a <code>String</code> value containing the name of the tag
     * @param attr a <code>String</code> value containing the tag attributes.
     * @param contents a <code>String</code> value with the contents of the tag.     
     * No checking of the code is carried out.     
     * @return a <code>String</code> value with the tag.
     */
    public static String tag(String name, String attr, String contents) {
	return tag(name,attr,contents,false);
    }

    /**
     * This version of the <code>tag</code> method  returns a generic non-ending HTML tag.
     * So far, this version <strong>cannot</strong> be used to generate a generic XML tag.
     *
     * @param name a <code>String</code> value containing the name of the tag
     * @param attr a <code>String</code> value containing the tag attributes.
     * @return a <code>String</code> value with the tag.
     */
    public static String tag(String name, String attr) {
	return "<"+name+" "+attr+">";
    }
    

    public static String html(String head, String body) {
	return tag("html","",
		   tag("head","",head)+
		   tag("body","",body)
		   );
    }    
    
    /**
     * The <code>script</code> method correspond with the HTML tag <code>script</code>.
     * If the script type is <code>text/javascript</code>, it is more convenient to
     * invoke the other {@link #script(String, String)  <code>script</code> method}.
     * 
     *
     * @param src a <code>String</code> value with the URL of file containig javascript code and
     * it will appear as the <code>src</code> attribute.
     * If it is an empty string, the <code>src</code> attribute does not appear. 
     * @param type a <code>String</code> value with the type of script, since it will 
     * usually be <code>"text/javascript"</code>.
     * @param contents a <code>String</code> value with the script (if any), it may be the ampty string.
     * No checking of the code is carried out.
     * @return a <code>String</code> value with the tag.
     */
    public static String script(String src, String type, String contents) {
	StringBuffer res = new StringBuffer("<script ");
	if (!isEmpty(src)) {
	    res.append("src=\""+src+"\"");
	}
	res.append("type=\""+type+"\">");
	res.append(contents);
	res.append("</script>\n\n");
	return res.toString();
	
    }
    /**
     * The <code>script</code> method for  javascript code.
     *
     * @param src a <code>String</code> value with the URL of file containig javascript code and
     * it will appear as the <code>src</code> attribute.
     * If it is an empty string, the <code>src</code> attribute does not appear. 
     * @param contents a <code>String</code> value with the script (if any), it may be the ampty string.
     * No checking of the code is carried out.     
     * @return a <code>String</code> value with the tag.
     */
    public static String script(String src, String contents) {
	return script(src,"text/javascript",contents);
    }
    
    /**
     * The <code>noscript</code> method correspond with the HTML tag <code>noscript</code>.
     *
     * @param contents a <code>String</code> value with the tag contents.
     * @return a <code>String</code> value with the tag.
     */
    public static String noscript(String contents){
	StringBuffer res = new StringBuffer("<noscript>\n");
	res.append(contents);
	res.append("</noscript>\n");
	return res.toString();
    }
    
    
    /**
     * The <code>col</code> method correspond with the HTML tag <code>col</code>.
     *
     * @param attr a <code>String</code> value containing the tag attributes.
     * @return a <code>String</code> value with the tag.
     */
    public static String col(String attr){
	return "<col "+attr+">\n";
    }
    
    
    /**
     * The <code>tr</code> method correspond with the HTML tag <code>tr</code>.
     *
     * @param attr a <code>String</code> value containing the tag attributes.
     * @param contents a <code>String</code> value with the tag contents.
     * @return a <code>String</code> value with the tag.
     */
    public static String tr(String attr, String contents) {
	return "<tr "+attr+">"+contents+"</tr>\n";
    }

    /**
     * The <code>th</code> method correspond with the HTML tag <code>th</code>.
     *
     * @param attr a <code>String</code> value containing the tag attributes.
     * @param contents a <code>String</code> value with the tag contents.
     * @return a <code>String</code> value with the tag.
     */
    public static String th(String attr, String contents) {
	return "<th "+attr+">"+contents+"</th>\n";
    }
    
    /**
     * The <code>td</code> method correspond with the HTML tag <code>td</code>.
     *
     * @param attr a <code>String</code> value containing the tag attributes.
     * @param contents a <code>String</code> value with the tag contents.
     * @return a <code>String</code> value with the tag.
     */
    public static String td(String attr, String contents) {
	return "<td "+attr+">"+contents+"</td>\n";
    }
    
    /**
     * The <code>caption</code> method correspond with the HTML tag <code>caption</code>.
     *
     * @param attr a <code>String</code> value containing the tag attributes.
     * @param contents a <code>String</code> value with the tag contents.
     * @return a <code>String</code> value with the tag.
     */
    public static String caption(String attr, String contents) {
	return "<caption "+attr+">"+contents+"</caption>\n";
    }
    
    /**
     * The <code>table</code> method correspond with the HTML tag <code>table</code>.
     *
     * @param attr a <code>String</code> value containing the tag attributes.
     * @param contents a <code>String</code> value with the tag contents.
     * @return a <code>String</code> value with the tag.
     */
    public static String table(String attr, String contents){
	return tag("table",attr,contents);
    }

    public static String dl(String attr, String contents) {
	return tag("dl",attr,contents);	
    }
    public static String dt(String attr, String contents) {
	return tag("dt",attr,contents);	
    }
    public static String dd(String attr, String contents) {
	return tag("dd",attr,contents);	
    }
    /**
     * The <code>br</code> method correspond with the HTML tag <code>br</code>.
     *
     * @param attr a <code>String</code> value containing the tag attributes.
     * @return a <code>String</code> value with the tag.
     */
    public static String br(String attr){
	return "<br "+attr+">\n";
    }    

    /**
     * The <code>div</code> method correspond with the HTML tag <code>div</code>.
     *
     * @param attr a <code>String</code> value containing the tag attributes.
     * @param contents a <code>String</code> value with the tag contents.
     * @return a <code>String</code> value with the tag.
     */
    public static String div(String attr, String contents){
	return tag("div",attr,contents);
    }
    
    public static String ul(String attr, String contents){
	return tag("ul",attr,contents);
    }
    
    public static String li(String attr, String contents){
	return tag("li",attr,contents);
    }
    
    /**
     * The <code>p</code> method correspond with the HTML tag <code>p</code>.
     *
     * @param attr a <code>String</code> value containing the tag attributes.
     * @param contents a <code>String</code> value with the tag contents.
     * @return a <code>String</code> value with the tag.
     */
    public static String p(String attr, String contents){
	return tag("p",attr,contents);
    }
    /**
     * The <code>h</code> method correspond with the HTML tag <code>p</code>.
     *
     * @param level a <code>int</code> with the header level.
     * @param attr a <code>String</code> value containing the tag attributes.
     * @param contents a <code>String</code> value with the tag contents.
     * @return a <code>String</code> value with the tag.
     */
    public static String h(int level, String attr, String contents){
	return tag("h"+level,attr,contents);
    }
    
    /**
     * The <code>span</code> method correspond with the HTML tag <code>span</code>.
     *
     * @param attr a <code>String</code> value containing the tag attributes.
     * @param contents a <code>String</code> value with the tag contents.
     * @return a <code>String</code> value with the tag.
     */
    public static String span(String attr, String contents){
	return tag("span",attr,contents);
    }
    
    /**
     * The <code>q</code> method correspond with the HTML tag <code>q</code>.
     *
     * @param attr a <code>String</code> value containing the tag attributes.
     * @param contents a <code>String</code> value with the tag contents.
     * @return a <code>String</code> value with the tag.
     */
    public static String q(String attr, String contents){
	return tag("q",attr,contents);
    }
    
    /**
     * The <code>b</code> method correspond with the HTML tag <code>b</code>.
     *
     * @param attr a <code>String</code> value containing the tag attributes.
     * @param contents a <code>String</code> value with the tag contents.
     * @return a <code>String</code> value with the tag.
     */
    public static String b(String attr, String contents){
	return tag("b",attr,contents);
    }
    
    /**
     * The <code>em</code> method correspond with the HTML tag <code>em</code>.
     *
     * @param attr a <code>String</code> value containing the tag attributes.
     * @param contents a <code>String</code> value with the tag contents.
     * @return a <code>String</code> value with the tag.
     */
    public static String em(String attr, String contents){
	return tag("em",attr,contents);
    }
    
    /**
     * The <code>button</code> method correspond with the HTML tag <code>button</code>.
     *
     * @param name a <code>String</code> value with the name attribute 
     * @param tipo a <code>String</code> value with the type attribute
     * @param valor a <code>String</code> value with the value attribute
     * @param attr a <code>String</code> value with extra attributes
     * @param contents a <code>String</code> value with the contents of the button
     * @return a <code>String</code> value
     */
    public static String button(String name, String tipo,
				String valor,
				String attr,
				String contents) {
	return tag("button",name(name)+type(tipo)+value(valor)+attr,contents);
    }
    /**
     * The <code>fieldset</code> method correspond with the HTML tag <code>fieldset</code>.
     *
     * @param label a <code>String</code> value with the label of the fieldset.
     * @param attr a <code>String</code> value with the tag attributes.
     * @param contents a <code>String</code> value  with the contents
     * @return a <code>String</code> value with the tag
     */
    public static String fieldset(String label, String attr, String contents){
	StringBuffer fieldSet = new StringBuffer();
	if (!isEmpty(contents)) {
	    fieldSet.append("<fieldset "+attr+"><legend>"+label+"</legend>\n");
	    fieldSet.append(contents);
	    fieldSet.append("</fieldset>\n");
	}
	return fieldSet.toString();
    }
    
    /**
     * The <code>form</code> method correspond with the HTML tag <code>form</code>.
     *
     * @param action a <code>String</code> value with the action attribute.
     * @param method a <code>String</code> value with the method attribute.
     * @param attr a <code>String</code> value with extra attributes.
     * @param contents a <code>String</code> value with the contents.
     * @return a <code>String</code> value whit the tag.
     */
    public static String form(String action, String method,
			      String attr, String contents) {
	return tag("form",attribute("action",action)+
		          attribute("method",method)+attr,
			  contents);
	
    }
    
    
    /**
     * The <code>form</code> method correspond with the HTML tag <code>form</code>.
     * The method is the default one.
     *
     * @param action a <code>String</code> value with the action attribute.
     * @param attr a <code>String</code> value with extra attributes.
     * @param contents a <code>String</code> value with the contents.
     * @return a <code>String</code> value whit the tag.
     */
    public static String form(String action, String attr, String contents){
	return form(action,getFormMethod(),attr,contents);
    }
    
    /**
     * The <code>a</code> method correspond with the HTML tag <code>a</code>.
     *
     * @param href a <code>String</code> value with the href attribute.
     * @param attr a <code>String</code> value containing the tag attributes.
     * @param contents a <code>String</code> value with the tag contents.
     * @return a <code>String</code> value with the tag.
     */
    public static String a(String href,
			   String attr,
			   String texto) {
        return 
            "<a href=\""+href+"\" "+attr+">" + texto+ "</a>\n";
    }
    
   /**
     * The <code>label</code> method correspond with the HTML tag <code>label</code>.
     *
     * @param forAttr a <code>String</code> value with the for attribute.
     * @param labelTxt a <code>String</code> value with the label text.
     * @param attr a <code>String</code> value with extra attributes
     * @return a <code>String</code> value with the tag.
     */
    public static String label(String forAttr, String labelTxt, 
			       String attr){
	return tag("label",attribute("for",forAttr)+attr,labelTxt);
    }
    
    /**
     * The <code>input</code> method correspond with the HTML tag <code>input</code>.
     *
     * @param id a <code>String</code> value with the id attribute.
     * @param name a <code>String</code> value with the name attribute.
     * @param attr a <code>String</code> value extra attributes
     * @return a <code>String</code> value with the tag.
     */
    public static String input(String id, String name, String attr) {
	return tag("input",attribute("id",id)+attribute("name",name)+attr);
    }
    
    /**
     * The <code>input</code> method correspond with the HTML tag <code>input</code> when
     * the attribute name and id have the same value.
     *
     * @param name a <code>String</code> value with the id and name attributes.
     * @param attr a <code>String</code> value with extra attributes
     * @return a <code>String</code> value with the tag.
     */
    public static String input(String name, String attr) {
	return input(name,name,attr);
    }
    /**
     * The <code>textarea</code> method correspond with the HTML tag <code>textarea</code> 
     *
     * @param id a <code>String</code> value with the id attribute.
     * @param name a <code>String</code> value with the name attribute.
     * @param tamX an <code>int</code> value with the rows  attribute.
     * @param tamY an <code>int</code> value with the cols  attribute.
     * @param value a <code>String</code> value with the value.
     * @param attr a <code>String</code> value with  extra attributes.
     * @return a <code>String</code> value value with the tag.
     */
    public static String textarea(String id, String name,
				  int tamX, int tamY,
				  String value,String attr) {
	return tag("textarea",
		   attribute("id",id)+attribute("name",name)+
		   attribute("rows",tamY+"")+attribute("cols",tamX+"")+attr,
		   value,true);
    }
    /**
     * The <code>textarea</code> method correspond with the HTML tag <code>textarea</code>  when
     * the attribute name and id have the same value.
     *
     * @param name a <code>String</code> value with the id and name attributes.
     * @param tamX an <code>int</code> value with the rows  attribute.
     * @param tamY an <code>int</code> value with the cols  attribute.
     * @param value a <code>String</code> value with the value.
     * @param attr a <code>String</code> value with  extra attributes.
     * @return a <code>String</code> value with the tag.

     */
    public static String textarea(String name,
				int tamX, int tamY,
				String value,String attr) {
	return textarea(name,name,tamX,tamY,value,attr);
    }
    /**
     * The <code>textarea</code> method correspond with the HTML tag <code>textarea</code>.
     * The options are given as a text.
     *
     * @param id a <code>String</code> value with the id attribute.
     * @param name a <code>String</code> value with the name attribute.
     * @param attr a <code>String</code> value with  extra attributes.
     * @param options a <code>String</code> value with the options
     * @return a <code>String</code> value with the tag.
     */
    public static String select(String id,
				String name, 
				String attr,
				String options) {
	return tag("select",attribute("id",id)+attribute("name",name)+attr,options);
    }
    /**
     * The <code>textarea</code> method correspond with the HTML tag <code>textarea</code>.
     * The options are given in a list.
     *
     * @param id a <code>String</code> value with the id attribute.
     * @param name a <code>String</code> value with the name attribute.
     * @param attr a <code>String</code> value with  extra attributes.
     * @param selectOptionList an <code>AbstractSequentialList</code> value containing 
     * list whose element classes are SelectOption.
     * @param valor a <code>String</code> with the value marked as selected.
     * @return a <code>String</code> value with the tag.
     */
    public static String select(String id,
				String name, 
				String attr,
				AbstractSequentialList selectOptionList,
				String valor ) {
	StringBuffer sb = new StringBuffer();
	for ( Iterator itr = selectOptionList.iterator(); itr.hasNext(); ) {
	    SelectOption op = (SelectOption)itr.next();
	    String valorOP=op.getValue();
	    String selected=null;
	    if ( valor!=null && valor.equals(valorOP) ) {
		selected="selected";
	    }
	    
	    sb.append(option(attribute("value",valorOP)+
			     attribute("selected",selected)+
			     op.getAttrs(),
			     op.getText()));
	}
        return select(id,name,attr,sb.toString());
    }
    /**
     * The <code>option</code> method correspond with the HTML tag <code>option</code>.
     *
     * @param attr a <code>String</code> value with the tag attributes.
     * @param text a <code>String</code> value with the text.
     * @return a <code>String</code> value with the tag.
     */
    public static String option(String attr,String text) {
	return tag("option",attr,text);
    }
    
    /**
     * The <code>inputRadio</code> method correspond with the HTML tag <code>option</code> whose type
     * is <code>radio</code>.
     *
     * @param id a <code>String</code> value with the id attribute.
     * @param name a <code>String</code> value with the name attribute.
     * @param _attr a <code>String</code> value with  extra attributes.
     * @param thisValue a <code>String</code> with the value attribute.
     * @param selectedValue a <code>String</code> value with selected value assiciated to
     * all radio inputs with the same name.
     * @return a <code>String</code> value
     */
    public static String inputRadio(String id, String name, 
				    String _attr,
				    String thisValue, 
				    String selectedValue) {
	String attr = _attr + type("radio") + value(thisValue);
	if ( thisValue.equals(selectedValue) ) {
	    attr = attr + " CHECKED ";
	}
	return input(id,name,attr);
    }
    
    /**************/
    /* Attributes */
    /**************/
    
        
    /**
     * The <code>attribute</code> includes an attribute. If the value is empty,
     * the empty string is returned.
     *
     * @param name a <code>String</code> value with the attribute name.
     * @param valor a <code>String</code> value with the value.
     * @return a <code>String</code> value with the attribute.
     */
    public static String attribute(String name, String valor){
        if ( isEmpty(valor) ) {
            return "";
        }
        return " "+name+"=\""+valor+"\" ";
    }
    /**
     * The <code>clase</code> method returns the class attribute. Since
     * class is already a reserved word in Java, it has is name in Spanish.
     *
     * @param value a <code>String</code> with the attribute value
     * @return a <code>String</code> value the attribute.
     */
    public static String clase(String value) {
	return attribute("class",value);
    }
    
    
    /**
     * The <code>accesskey</code> method returns the accesskey attribute.
     *
     * @param c a <code>char</code> value with the key.
     * @return a <code>String</code> value with the attribute.
     */
    public static String accesskey(char c) {
	return attribute("accesskey",c+"");
    }
    /**
     * The <code>acceptCharset</code>  method returns the acceptCharset attribute.
     *
     * @param s a <code>String</code> is the value
     * @return a <code>String</code>  value with the attribute.
     */
    public static String acceptCharset(String s) {
	return attribute("accept-charset",s);
    }
    /**
     * The <code>title</code>  method returns the title attribute.
     *
     * @param s a <code>String</code> with the attribute value.
     * @return a <code>String</code> value with the attribute.
     */
    public static String title(String s) {
	return attribute("title",s);
    }
    /**
     * The <code>id</code>  method returns the id attribute.
     *
     * @param s a <code>String</code> with the attribute value.
     * @return a <code>String</code> value with the attribute.
     */
    public static String id(String s) {
	return attribute("id",s);
    }
    /**
     * The <code>value</code>  method returns the value attribute.
     *
     * @param s a <code>String</code> with the attribute value.
     * @return a <code>String</code> value with the attribute.
     */
    public static String value(String s) {
	return attribute("value",s);
    }
    /**
     * The <code>name</code>  method returns the name attribute.
     *
     * @param s a <code>String</code> with the attribute value.
     * @return a <code>String</code> value with the attribute.
     */
    public static String name(String s) {
	return attribute("name",s);
    }
    /**
     * The <code>style</code>  method returns the style attribute.
     *
     * @param s a <code>String</code> with the attribute value.
     * @return a <code>String</code> value with the attribute.
     */
    public static String style(String s) {
	return attribute("style",s);
    }
    /**
     * The <code>size</code>  method returns the size attribute.
     *
     * @param s a <code>String</code> with the attribute value.
     * @return a <code>String</code> value with the attribute.
     */
    public static String size(String s) {
	return attribute("size",s);
    }
    
    /**
     * The <code>type</code>  method returns the type attribute.
     *
     * @param s a <code>String</code> with the attribute value.
     * @return a <code>String</code> value  with the attribute.
     */
    public static String type(String s) {
	return attribute("type",s);
    }
    /**
     * The <code>tabindex</code> method builds the tabindex attribute.
     *
     * @param s a <code>String</code> with the attribute value
     * @return a <code>String</code> value  with the attribute.
     */
    public static String tabindex(String s) {
	return attribute("tabindex",s);
    }

    
    /**
     * The <code>isEmpty</code> method returns true if
     * the string is not null and not empty.
     *
     * @param s a <code>String</code> value the string to be checked
     * @return a <code>boolean</code> value
     */
    private static boolean isEmpty(String s) {
	return s==null || s.trim().equals("");
    }

    /**
     * Describe <code>ponFoco</code> method here.
     *
     * @param name a <code>String</code> value
     * @return a <code>String</code> value
     */
    public static String ponFoco(String name) {
	return attribute("onClick","ponInputFoco('"+name+"')");
    }
    
    
    
} // HTML
